package com.example.servicedemo;

// All the Imports we need.
import android.app.Activity;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.Button;
import android.widget.TextView;

import com.example.servicedemo.services.RandomValueCallback;
import com.example.servicedemo.services.RandomValueService;
import com.example.servicedemo.services.RandomValueService.RandomValueBinder;

// My Activity
public class MainActivity extends Activity implements RandomValueCallback {

	// Some instance variables we need.
	private RandomValueService service = null;
	private MyServiceConnection connection = new MyServiceConnection(this);
	private TextView message;
	
	/* Important one since the service will act in a background thread and therefore the callback
	 * method will be executed in a background thread as well. But we don't have access to the UI
	 * elements from the background. So we need some reference to the UI thread.
	 */
	private Handler uiHandle = new Handler();
	
	@Override
	protected void onCreate(Bundle savedInstanceState) {
		// Initializing stuff.
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_main);
		Button toggleServiceButton = (Button)findViewById(R.id.toggle_service_button);
		toggleServiceButton.setOnClickListener(new ToggleServiceOnClickListener());
		
		message = (TextView)findViewById(R.id.message_text);
		
		Button stringButton = (Button)findViewById(R.id.getString);
		stringButton.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				// Just for fun: if the service is running we can manually request a new string.
				if(isServiceRunning()) {
					message.setText(service.getPseudoValueString());
				}
				else {
					message.setText("Service not running.");
				}
			}
		});
	}
	
	// The onClick Listener for the Start/Stop button
	class ToggleServiceOnClickListener implements View.OnClickListener {
		@Override
		public void onClick(View clickedView) {
			Button clickedButton = null;
			
			if(clickedView instanceof Button) {
				// Clicked on a button.
				clickedButton = (Button)clickedView;
			} 
			else {
				// Not a button? Get out of here before something crashes.
				return;
			}
			
			if(isServiceRunning()) {
				// To disconnect we have to unbind the service.
				unbindService(connection);
				
				// Changing the label. Did you guess it? ;)
				clickedButton.setText(R.string.start_service);
				
				/* Since the onServiceDisconnect callback is only called when something crashes,
				 * we manually reset the service variable.
				 */
				service = null;
			}
			else {
				// A specified Intent is needed in order to bind to the service.
				Intent serviceIntent = new Intent(clickedButton.getContext(), RandomValueService.class);
				bindService(serviceIntent, connection, Context.BIND_AUTO_CREATE);
				clickedButton.setText(R.string.stop_service);
			}
		}
	}
	
	// Just checking if the service is running.
	private boolean isServiceRunning() {
		return (service != null);
	}
	
	// Callback method for the running service.
	@Override
	public void didGatherNewStringValue(IBinder sender, final String newValue) {
		// Here we wrap around the method in a runnable posted to the UI thread.
		uiHandle.post(new Runnable() {
			@Override
			public void run() {
				updateDisplayedMessage(newValue);
			}
		});
	}

	// Well, does what it says.
	private void updateDisplayedMessage(String text) {
		message.setText(text);
	}
	
	// Own implementation of the ServiceConnection callback methods, because:
	class MyServiceConnection implements ServiceConnection  {
		// we need an instance variable to hold our Service Callback reference
		private RandomValueCallback cb = null;
		// as well as a constructor which needs this callback reference.
		public MyServiceConnection(RandomValueCallback callback) {
			cb = callback;
		}
		
		@Override
		public void onServiceDisconnected(ComponentName name) {
			// Only called when crashed or killed.
			service = null;
		}
		
		@Override
		public void onServiceConnected(ComponentName name, IBinder boundService) {
			// Service was bound. So we'll save the reference for further use
			RandomValueBinder binder = (RandomValueBinder)boundService;		
			service = binder.getService();
			// and provide the callback object.
			service.setCallbackListener(cb);
		}
	};
	
}
